﻿# PowerShell Temizlik Betiği
# Hazırlayan: JetEkip

# Yönetici Yetkisi Kontrolü
If (-Not ([Security.Principal.WindowsPrincipal] [Security.Principal.WindowsIdentity]::GetCurrent()).IsInRole([Security.Principal.WindowsBuiltInRole] "Administrator")) {
    # Betik yönetici olarak çalışmıyorsa kendini yönetici olarak tekrar çalıştırır
    Start-Process powershell.exe "-ExecutionPolicy Bypass -File `"$PSCommandPath`"" -Verb RunAs
    Exit
}

# Execution Policy'i Unrestricted Olarak Ayarla
Try {
    Set-ExecutionPolicy Unrestricted -Scope CurrentUser -Force
    Write-Host "Execution Policy 'Unrestricted' olarak ayarlandı."
} Catch {
    Write-Host "Execution Policy ayarlanamadı. Yönetici yetkisine sahip olup olmadığınızı kontrol edin."
    Exit
}

# UTF-8 Kodlaması Aktif Ediliyor
[Console]::OutputEncoding = [System.Text.Encoding]::UTF8

# Başlangıç: Temizlik Log Dosyası
$logFile = "$($env:USERPROFILE)\Desktop\cleaning_log_$(Get-Date -Format 'yyyyMMddHHmmss').txt"
$startTime = Get-Date
"Temizlik Başladı: $startTime" | Out-File -FilePath $logFile -Append
"Bu temizlik betiği JetEkip tarafından oluşturulmuştur." | Out-File -FilePath $logFile -Append

# İlerleme Çubuğu için Gerekli Değişkenler
$totalSteps = 21  # Adım sayısı güncellendi
$currentStep = 0  # Şu anki adım

# Fonksiyon: İlerleme Çubuğunu Gösterme
function Show-ProgressBar {
    param (
        [string]$message
    )
    $percent = [math]::Round(($currentStep / $totalSteps) * 100)
    Write-Progress -Activity "JetEkip Temizlik Betiği" -Status "$message ($percent%)" -PercentComplete $percent
    $currentStep++
}

# Fonksiyon: Klasör Temizleme
function Clean-Folder {
    param (
        [string]$folderPath
    )

    Show-ProgressBar -message "Klasör Temizleniyor: $folderPath"

    if (Test-Path $folderPath) {
        try {
            Remove-Item -Path "$folderPath\*" -Recurse -Force -ErrorAction Stop
            Write-Output "Temizlendi: $folderPath" | Out-File -FilePath $logFile -Append
        } catch {
            Write-Output "Hata: $folderPath silinirken bir sorun oluştu." | Out-File -FilePath $logFile -Append
        }
    } else {
        Write-Output "Uyarı: $folderPath bulunamadı." | Out-File -FilePath $logFile -Append
    }
}

# Modül 1: Temp Klasörlerini Temizleme
Clean-Folder "C:\Windows\Temp"
Clean-Folder "$env:TEMP"
Clean-Folder "$env:LOCALAPPDATA\Temp"
Clean-Folder "$env:LOCALAPPDATA\Microsoft\Windows\INetCache"

# Modül 2: Prefetch ve Diğer Sistem Klasörleri
Clean-Folder "C:\Windows\Prefetch"
Clean-Folder "C:\WINDOWS\SoftwareDistribution"
Clean-Folder "C:\ProgramData\Microsoft\Search\Data\Applications\Windows"
Clean-Folder "C:\WINDOWS\ccmcache"
Clean-Folder "C:\ProgramData\Riverbed\Steelhead_Mobile\Datastore"
Clean-Folder "C:\MSOCache"
Clean-Folder "C:\Windows\CCM\Temp"

# Servisleri Durdurma ve Temizlik
function Stop-And-Clean-Service {
    param (
        [string]$serviceName,
        [string]$serviceFolderPath
    )

    Show-ProgressBar -message "Servis Durduruluyor: $serviceName"

    # Servisi Durdur
    try {
        Stop-Service -Name $serviceName -Force -ErrorAction SilentlyContinue
        Write-Output "$serviceName durduruldu." | Out-File -FilePath $logFile -Append
    } catch {
        Write-Output "$serviceName durdurulamadı. Zaten durdurulmuş olabilir." | Out-File -FilePath $logFile -Append
    }

    # Klasörü Temizle
    Clean-Folder $serviceFolderPath

    # Servisi Tekrar Başlat (İsteğe Bağlı)
    try {
        Start-Service -Name $serviceName -ErrorAction SilentlyContinue
        Write-Output "$serviceName yeniden başlatıldı." | Out-File -FilePath $logFile -Append
    } catch {
        Write-Output "$serviceName yeniden başlatılamadı." | Out-File -FilePath $logFile -Append
    }
}

# Windows Search Servisi
Stop-And-Clean-Service "WSearch" "C:\ProgramData\Microsoft\Search\Data\Applications\Windows"

# Windows Update Servisi
Stop-And-Clean-Service "wuauserv" "C:\WINDOWs\SoftwareDistribution"

# Geri Dönüşüm Kutusu Temizleme
function Clean-RecycleBin {
    Show-ProgressBar -message "Geri Dönüşüm Kutusu Temizleniyor"
    try {
        Clear-RecycleBin -Force -ErrorAction SilentlyContinue
        Write-Output "Geri Dönüşüm Kutusu temizlendi." | Out-File -FilePath $logFile -Append
    } catch {
        Write-Output "Geri Dönüşüm Kutusu temizlenirken bir hata oluştu." | Out-File -FilePath $logFile -Append
    }
}

Clean-RecycleBin

# Windows Event Log Temizleme
function Clean-EventLogs {
    $eventLogs = @("Application", "System", "Security")
    foreach ($log in $eventLogs) {
        Show-ProgressBar -message "Event Log Temizleniyor: $log"
        try {
            wevtutil cl $log
            Write-Output "$log Event Log temizlendi." | Out-File -FilePath $logFile -Append
        } catch {
            Write-Output "$log Event Log temizlenirken bir hata oluştu." | Out-File -FilePath $logFile -Append
        }
    }
}

Clean-EventLogs

# Disk Temizleme Aracının Entegre Edilmesi (Popup Engellendi)
Show-ProgressBar -message "Disk Temizleme Aracı Çalıştırılıyor"
Start-Process cleanmgr.exe -ArgumentList "/verylowdisk /autoclean" -Wait
Write-Output "Disk Temizleme Aracı çalıştırıldı." | Out-File -FilePath $logFile -Append

# Eski Windows Güncellemelerini Temizleme
function Clean-WindowsUpdateBackup {
    Show-ProgressBar -message "Eski Windows Güncellemeleri Temizleniyor"
    Dism.exe /online /cleanup-image /startcomponentcleanup /quiet | Out-Null
    Write-Output "Eski Windows güncelleme dosyaları temizlendi." | Out-File -FilePath $logFile -Append
}

Clean-WindowsUpdateBackup

# Internet Explorer/Edge Önbelleği Temizleme
function Clean-IECache {
    Show-ProgressBar -message "Internet Explorer/Edge Önbelleği Temizleniyor"
    RunDll32.exe InetCpl.cpl,ClearMyTracksByProcess 255
    Write-Output "Internet Explorer/Edge önbelleği temizlendi." | Out-File -FilePath $logFile -Append
}

Clean-IECache

# Google Chrome Önbelleği Temizleme
function Clean-ChromeCache {
    Show-ProgressBar -message "Google Chrome Önbelleği Temizleniyor"
    $chromeCachePath = "$env:LOCALAPPDATA\Google\Chrome\User Data\Default\Cache"
    Clean-Folder $chromeCachePath
}

Clean-ChromeCache

# Eski Geri Yükleme Noktalarını Temizleme
function Clean-SystemRestorePoints {
    Show-ProgressBar -message "Eski Geri Yükleme Noktaları Temizleniyor"
    vssadmin delete shadows /for=C: /olderthan=1day /quiet | Out-Null
    Write-Output "Eski geri yükleme noktaları temizlendi." | Out-File -FilePath $logFile -Append
}

Clean-SystemRestorePoints

# Hızlı Erişim ve Son Dosyalar Geçmişini Temizleme
function Clean-QuickAccessHistory {
    Show-ProgressBar -message "Hızlı Erişim ve Son Kullanılanlar Temizleniyor"
    Remove-Item "$env:APPDATA\Microsoft\Windows\Recent\*" -Recurse -Force -ErrorAction SilentlyContinue
    Write-Output "Hızlı Erişim ve Son Kullanılanlar geçmişi temizlendi." | Out-File -FilePath $logFile -Append
}

Clean-QuickAccessHistory

# DNS Önbelleğini Temizleme
function Clear-DnsCache {
    Show-ProgressBar -message "DNS Önbelleği Temizleniyor"
    ipconfig /flushdns
    Write-Output "DNS Önbelleği temizlendi." | Out-File -FilePath $logFile -Append
}

Clear-DnsCache

# Temizlik Bitti: Log Dosyası Güncelleme
$endTime = Get-Date
"Temizlik Tamamlandı: $endTime" | Out-File -FilePath $logFile -Append
$duration = $endTime - $startTime
"Toplam Süre: $($duration.Hours) saat, $($duration.Minutes) dakika, $($duration.Seconds) saniye" | Out-File -FilePath $logFile -Append

# Sistem Yeniden Başlatma Bilgisi ve Öneriler
function Show-SystemRecommendations {
    # Bilgisayar ne kadar süredir açık?
    $uptime = (Get-Date) - (gcim Win32_OperatingSystem).LastBootUpTime
    $uptimeMessage = "Bilgisayarınız yaklaşık {0} gün {1} saat {2} dakika boyunca açık." -f $uptime.Days, $uptime.Hours, $uptime.Minutes

    # Disklerin Sağlık Durumu
    $diskInfo = Get-PhysicalDisk | Select-Object DeviceID, MediaType, OperationalStatus
    $diskHealthMessage = "Bilgisayardaki diskler ve sağlık durumları aşağıdaki gibidir:`n"
    foreach ($disk in $diskInfo) {
        $status = if ($disk.OperationalStatus -eq "OK") { "Sağlıklı" } else { "HATA TESPİT EDİLDİ" }
        $diskHealthMessage += "Disk: $($disk.DeviceID) - Durum: $status - Tür: $($disk.MediaType)`n"
    }

    # Bilgilendirme Mesajını Gösterme (Modern Görünüm)
    $recommendationMessage = @"
===============================
       Sistem Durum Raporu
===============================

$uptimeMessage

$diskHealthMessage

===============================
"@

    Add-Type -AssemblyName PresentationFramework
    [System.Windows.MessageBox]::Show($recommendationMessage, "JetEkip Temizlik Bilgilendirme", [System.Windows.MessageBoxButton]::OK, [System.Windows.MessageBoxImage]::Information)
}

# Temizlik Sonrası Sistem Önerilerini Gösterme
Show-SystemRecommendations

# Temizlik Tamamlandığında Bildirim Gönderme
[System.Media.SystemSounds]::Asterisk.Play()
Add-Type -AssemblyName PresentationFramework
[System.Windows.MessageBox]::Show('Temizlik Tamamlandı! Betik yazarı: JetEkip')

# Sistem Yeniden Başlatma (İsteğe Bağlı)
$restartChoice = Read-Host "Temizlik tamamlandı. Bilgisayarı yeniden başlatmak istiyor musunuz? (E/H)"
if ($restartChoice.ToUpper() -eq "E") {
    Restart-Computer
}
